<?php
/**
 * Plugin Name: FG Magento to WooCommerce Premium Tiered Prices module
 * Depends:		FG Magento to WooCommerce Premium
 * Depends:		Tiered Prices
 * Plugin Uri:  https://www.fredericgilles.net/fg-magento-to-woocommerce/
 * Description: This add-on will import the Magento tiered prices
 * 				Needs the «FG Magento to WooCommerce Premium» plugin
 *              Needs the «WooCommerce Tiered Price Table» plugin
 * Version:     1.0.0
 * Author:      Frédéric GILLES
 */

// Exit if accessed directly
if ( !defined( 'ABSPATH' ) ) exit;

add_action( 'admin_init', 'fgm2wc_tiered_prices_test_requirements' );

if ( !function_exists( 'fgm2wc_tiered_prices_test_requirements' ) ) {
	function fgm2wc_tiered_prices_test_requirements() {
		new fgm2wc_tiered_prices_requirements();
	}
}

if ( !class_exists('fgm2wc_tiered_prices_requirements', false) ) {
	class fgm2wc_tiered_prices_requirements {
		private $parent_plugin = 'fg-magento-to-woocommerce-premium/fg-magento-to-woocommerce-premium.php';
		private $required_premium_version = '2.95.0';

		public function __construct() {
			load_plugin_textdomain( 'fgm2wc_tiered_prices', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
			if ( !is_plugin_active($this->parent_plugin) ) {
				add_action( 'admin_notices', array($this, 'fgm2wc_tiered_prices_error') );
			} else {
				$plugin_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $this->parent_plugin);
				if ( !$plugin_data or version_compare($plugin_data['Version'], $this->required_premium_version, '<') ) {
					add_action( 'admin_notices', array($this, 'fgm2wc_tiered_prices_version_error') );
				}
			}
		}
		
		/**
		 * Print an error message if the Premium plugin is not activated
		 */
		function fgm2wc_tiered_prices_error() {
			echo '<div class="error"><p>[fgm2wc_tiered_prices] '.__('The Tiered Prices module needs the «FG Magento to WooCommerce Premium» plugin to work. Please install and activate <strong>FG Magento to WooCommerce Premium</strong>.', 'fgm2wc_tiered_prices').'<br /><a href="https://www.fredericgilles.net/fg-magento-to-woocommerce/" target="_blank">https://www.fredericgilles.net/fg-magento-to-woocommerce/</a></p></div>';
		}
		
		/**
		 * Print an error message if the Premium plugin is not at the required version
		 */
		function fgm2wc_tiered_prices_version_error() {
			printf('<div class="error"><p>[fgm2wc_tiered_prices] '.__('The Tiered Prices module needs at least the <strong>version %s</strong> of the «FG Magento to WooCommerce Premium» plugin to work. Please install and activate <strong>FG Magento to WooCommerce Premium</strong> at least the <strong>version %s</strong>.', 'fgm2wc_tiered_prices').'<br /><a href="https://www.fredericgilles.net/fg-magento-to-woocommerce/" target="_blank">https://www.fredericgilles.net/fg-magento-to-woocommerce/</a></p></div>', $this->required_premium_version, $this->required_premium_version);
		}
		
	}
}

if ( !defined('WP_LOAD_IMPORTERS') && !defined('DOING_AJAX') ) return;

add_action( 'plugins_loaded', 'fgm2wc_tiered_prices_load', 25 );

if ( !function_exists( 'fgm2wc_tiered_prices_load' ) ) {
	function fgm2wc_tiered_prices_load() {
		if ( !defined('FGM2WCP_LOADED') ) return;

		load_plugin_textdomain( 'fgm2wc_tiered_prices', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
		
		global $fgm2wcp;
		new fgm2wc_tiered_prices($fgm2wcp);
	}
}

if ( !class_exists('fgm2wc_tiered_prices', false) ) {
	class fgm2wc_tiered_prices {
		
		private $plugin;
		
		/**
		 * Sets up the plugin
		 *
		 */
		public function __construct($plugin) {
			
			$this->plugin = $plugin;
			
			add_filter( 'fgm2wc_pre_display_admin_page', array ($this, 'process_admin_page'), 11, 1 );
			add_action( 'fgm2wc_post_insert_product', array ($this, 'import_product_tier_prices'), 10, 2 );
		}
		
		/**
		 * Add information to the admin page
		 * 
		 * @param array $data
		 * @return array
		 */
		public function process_admin_page($data) {
			$data['title'] .= ' ' . __('+ Tiered Prices module', __CLASS__);
			$data['description'] .= "<br />" . sprintf(__('The Tiered Prices module will also import the Magento tiered prices. The <a href="%s" target="_blank">WooCommerce Tiered Price Table</a> plugin is required.', __CLASS__), 'https://wordpress.org/plugins/tier-pricing-table/');
			return $data;
		}
		
		/**
		 * Import the product tiered prices
		 * 
		 * @param int $new_post_id WP post ID
		 * @param array $product Product data
		 */
		public function import_product_tier_prices($new_post_id, $product) {
			$tier_prices = $this->get_tier_prices($product['entity_id']);
			if ( !empty($tier_prices) ) {
				add_post_meta($new_post_id, '_fixed_price_rules', $tier_prices);
				add_post_meta($new_post_id, '_tiered_price_rules_type', 'fixed');
			}
		}
		
		/**
		 * Get the tier prices of a product
		 * 
		 * @param int $product_id Product ID
		 * @return array Tier prices
		 */
		private function get_tier_prices($product_id) {
			$prices = array();
			$prefix = $this->plugin->plugin_options['prefix'];
			
			$sql = "
				SELECT tp.qty, tp.value
				FROM ${prefix}catalog_product_entity_tier_price tp
				WHERE tp.entity_id = '$product_id'
				AND tp.customer_group_id = 0
			";
			$result = $this->plugin->magento_query($sql);
			
			foreach ( $result as $row ) {
				$prices[intval($row['qty'])] = floatval($row['value']);
			}
			
			return $prices;
		}
		
	}
}
