<?php
/**
 * Plugin Name: FG Magento to WooCommerce Premium Customer Groups module
 * Depends:		FG Magento to WooCommerce Premium
 * Depends:		Groups
 * Depends:		WooCommerce Wholesale Prices
 * Plugin Uri:  https://www.fredericgilles.net/fg-magento-to-woocommerce/
 * Description: This add-on will import the Magento customer groups to WordPress user groups
 * 				Needs the «FG Magento to WooCommerce Premium» plugin
 *              Needs either the «Groups» or the «WooCommerce Wholesale Prices» plugins
 * Version:     1.4.0
 * Author:      Frédéric GILLES
 */

// Exit if accessed directly
if ( !defined( 'ABSPATH' ) ) exit;

add_action( 'admin_init', 'fgm2wc_customer_groups_test_requirements' );

if ( !function_exists( 'fgm2wc_customer_groups_test_requirements' ) ) {
	function fgm2wc_customer_groups_test_requirements() {
		new fgm2wc_customer_groups_requirements();
	}
}

if ( !class_exists('fgm2wc_customer_groups_requirements', false) ) {
	class fgm2wc_customer_groups_requirements {
		private $parent_plugin = 'fg-magento-to-woocommerce-premium/fg-magento-to-woocommerce-premium.php';
		private $required_premium_version = '2.66.0';

		public function __construct() {
			load_plugin_textdomain( 'fgm2wc_customer_groups', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
			if ( !is_plugin_active($this->parent_plugin) ) {
				add_action( 'admin_notices', array($this, 'fgm2wc_customer_groups_error') );
			} else {
				$plugin_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $this->parent_plugin);
				if ( !$plugin_data or version_compare($plugin_data['Version'], $this->required_premium_version, '<') ) {
					add_action( 'admin_notices', array($this, 'fgm2wc_customer_groups_version_error') );
				}
			}
		}
		
		/**
		 * Print an error message if the Premium plugin is not activated
		 */
		function fgm2wc_customer_groups_error() {
			echo '<div class="error"><p>[fgm2wc_customer_groups] '.__('The User Groups module needs the «FG Magento to WooCommerce Premium» plugin to work. Please install and activate <strong>FG Magento to WooCommerce Premium</strong>.', 'fgm2wc_customer_groups').'<br /><a href="https://www.fredericgilles.net/fg-magento-to-woocommerce/" target="_blank">https://www.fredericgilles.net/fg-magento-to-woocommerce/</a></p></div>';
		}
		
		/**
		 * Print an error message if the Premium plugin is not at the required version
		 */
		function fgm2wc_customer_groups_version_error() {
			printf('<div class="error"><p>[fgm2wc_customer_groups] '.__('The User Groups module needs at least the <strong>version %s</strong> of the «FG Magento to WooCommerce Premium» plugin to work. Please install and activate <strong>FG Magento to WooCommerce Premium</strong> at least the <strong>version %s</strong>.', 'fgm2wc_customer_groups').'<br /><a href="https://www.fredericgilles.net/fg-magento-to-woocommerce/" target="_blank">https://www.fredericgilles.net/fg-magento-to-woocommerce/</a></p></div>', $this->required_premium_version, $this->required_premium_version);
		}
		
	}
}

if ( !defined('WP_LOAD_IMPORTERS') && !defined('DOING_AJAX') ) return;

add_action( 'plugins_loaded', 'fgm2wc_customer_groups_load', 25 );

if ( !function_exists( 'fgm2wc_customer_groups_load' ) ) {
	function fgm2wc_customer_groups_load() {
		if ( !defined('FGM2WCP_LOADED') ) return;

		load_plugin_textdomain( 'fgm2wc_customer_groups', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
		
		global $fgm2wcp;
		new fgm2wc_customer_groups($fgm2wcp);
	}
}

if ( !class_exists('fgm2wc_customer_groups', false) ) {
	class fgm2wc_customer_groups {
		
		public $imported_groups = array();
		public $imported_roles = array();
		
		private $customer_groups_options = array();
		private $imported_products = array();
		
		/**
		 * Sets up the plugin
		 *
		 */
		public function __construct($plugin) {
			
			$this->plugin = $plugin;
			
			add_filter( 'fgm2wc_pre_display_admin_page', array ($this, 'process_admin_page'), 11, 1 );
			add_action( 'fgm2wc_post_display_behavior_options', array ($this, 'display_options') );
			add_action( 'fgm2wc_post_save_plugin_options', array ($this, 'save_options') );
			add_action( 'fgm2wc_post_empty_database', array ($this, 'delete_groups') );
			add_action( 'fgm2wc_post_import', array ($this, 'import_groups'), 9 ); // Import the groups before the customers
			add_action( 'fgm2wcp_post_add_customer', array ($this, 'import_customer_groups'), 10, 2 );
			
			// Default values
			$this->customer_groups_options = array(
				'import_groups' => 'as_roles',
				'import_group_prices' => 'from_tier_prices',
			);
			$options = get_option('fgm2wc_customer_groups_options');
			if ( is_array($options) ) {
				$this->customer_groups_options = array_merge($this->customer_groups_options, $options);
			}
		}
		
		/**
		 * Add information to the admin page
		 * 
		 * @param array $data
		 * @return array
		 */
		public function process_admin_page($data) {
			$data['title'] .= ' ' . __('+ Customer Groups module', __CLASS__);
			$data['description'] .= "<br />" . sprintf(__('The Customer Groups module will also import the Magento customer groups. Either the <a href="%s" target="_blank">Groups</a> plugin or the <a href="%s" target="_blank">WooCommerce Wholesale Prices</a> plugin is required.', __CLASS__), 'https://wordpress.org/plugins/groups/', 'https://wordpress.org/plugins/woocommerce-wholesale-prices/');
			return $data;
		}
		
		/**
		 * Add the customer groups options to the admin page
		 * 
		 * @since 1.1.0
		 */
		public function display_options() {
			echo '<tr><th>' . __('Customer groups:', __CLASS__) . '</th><td>';
			echo '<input type="radio" name="import_groups" id="import_groups_as_groups" value="as_groups" ' . checked($this->customer_groups_options['import_groups'], 'as_groups', false) . ' /> <label for="import_groups_as_groups">' . __('Import the customer groups as WordPress groups', __CLASS__) . '</label>';
			echo ' <small>- ' . sprintf(__('The <a href="%s" target="_blank">Groups</a> plugin is required.', __CLASS__), 'https://wordpress.org/plugins/groups/') . '</small><br />';
			echo '<input type="radio" name="import_groups" id="import_groups_as_roles" value="as_roles" ' . checked($this->customer_groups_options['import_groups'], 'as_roles', false) . ' /> <label for="import_groups_as_roles">' . __('Import the customer groups as WordPress roles', __CLASS__) . '</label>';
			echo ' <small>- ' . sprintf(__('Compatible with the <a href="%s" target="_blank">WooCommerce Wholesale Prices</a> plugin (optional)', __CLASS__), 'https://wordpress.org/plugins/woocommerce-wholesale-prices/') . '</small><br />';
			echo '</td></tr>';
			
			echo '<tr><th>' . __('Customer group prices:', __CLASS__) . '</th><td>';
			echo __('Import the customer group prices:', __CLASS__);
			echo ' <input type="radio" name="import_group_prices" id="import_group_prices_from_tier_prices" value="from_tier_prices" ' . checked($this->customer_groups_options['import_group_prices'], 'from_tier_prices', false) . ' /><label for="import_group_prices_from_tier_prices">' . __('from tier prices', __CLASS__) . '</label>&nbsp; ';
			echo '<input type="radio" name="import_group_prices" id="import_group_prices_from_group_prices" value="from_group_prices" ' . checked($this->customer_groups_options['import_group_prices'], 'from_group_prices', false) . ' /><label for="import_group_prices_from_group_prices">' . __('from group prices (Magento 1 only)', __CLASS__) . '</label>';
			echo '</td></tr>';
		}
			
		/**
		 * Save the Customer Groups options
		 *
		 * @since 1.1.0
		 */
		public function save_options() {
			$this->customer_groups_options = array_merge($this->customer_groups_options, $this->validate_form_info());
			update_option('fgm2wc_customer_groups_options', $this->customer_groups_options);
		}
		
		/**
		 * Validate POST info
		 *
		 * @since 1.1.0
		 * 
		 * @return array Form parameters
		 */
		private function validate_form_info() {
			$import_groups = filter_input(INPUT_POST, 'import_groups', FILTER_SANITIZE_STRING);
			$import_group_prices = filter_input(INPUT_POST, 'import_group_prices', FILTER_SANITIZE_STRING);
			return array(
				'import_groups' => !empty($import_groups)? $import_groups : 'as_roles',
				'import_group_prices' => !empty($import_group_prices)? $import_group_prices : 'from_tier_prices',
			);
		}
		
		/**
		 * Delete the groups
		 * 
		 */
		public function delete_groups() {
			global $wpdb;
			
			$sql_queries = array();
			$sql_queries[] = "TRUNCATE {$wpdb->prefix}groups_capability";
			$sql_queries[] = "TRUNCATE {$wpdb->prefix}groups_group";
			$sql_queries[] = "TRUNCATE {$wpdb->prefix}groups_group_capability";
			$sql_queries[] = "TRUNCATE {$wpdb->prefix}groups_user_capability";
			$sql_queries[] = "TRUNCATE {$wpdb->prefix}groups_user_group";
			foreach ( $sql_queries as $sql ) {
				$wpdb->query($sql);
			}
		}
		
		/**
		 * Import the groups
		 * 
		 * @since 1.1.0
		 */
		public function import_groups() {
			if ( $this->customer_groups_options['import_groups'] == 'as_groups' ) {
				$this->import_groups_as_groups();
			} else {
				$this->import_groups_as_roles();
			}
		}
		
		/**
		 * Import the groups as WordPress groups
		 * 
		 * @uses Groups_User_Group (Groups plugin)
		 */
		private function import_groups_as_groups() {
			if ( class_exists('Groups_Group') && class_exists('Groups_User_Group') ) {
				$this->imported_groups = array();
				$groups_count = 0;
				$groups = $this->get_groups();
				foreach ( $groups as $group ) {
					$new_group = Groups_Group::read_by_name($group['customer_group_code']);
					if ( $new_group !== false ) {
						$new_group_id = $new_group->group_id;
					} else {
						$new_group_id = Groups_Group::create(array(
							'name'		=> $group['customer_group_code'],
						));
					}
					$groups_count++;
					$this->imported_groups[$group['customer_group_id']] = $new_group_id;
				}
				$this->plugin->display_admin_notice(sprintf(_n('%d customer group imported', '%d customer groups imported', $groups_count, __CLASS__), $groups_count));
			}
		}
		
		/**
		 * Import the groups as WordPress user roles
		 * 
		 * @since 1.1.0
		 */
		private function import_groups_as_roles() {
			$this->imported_products = $this->plugin->get_imported_magento_products(true);
			$this->imported_roles = array();
			$roles_count = 0;
			$groups = $this->get_groups();
			foreach ( $groups as $group ) {
				if ( $group['customer_group_id'] > 1 ) { // Don't import "NOT LOGGED IN" and "General" groups
					$role_name = $group['customer_group_code'];
					$role_slug = sanitize_key(str_replace(' ', '_', $role_name));
					add_role($role_slug, $role_name, array(
						'read' => true,
						'have_wholesale_price' => true,
					));
					$roles_count++;
					$this->imported_roles[$group['customer_group_id']] = $role_slug;

					// Compatibility with WooCommerce Wholesale Prices
					if ( class_exists('WWP_Wholesale_Roles', false) ) {
						$wholesale_role = new WWP_Wholesale_Roles();
						$wholesale_role->registerCustomRole($role_slug, $role_name, array('desc' => ''));
						
						// Import the wholesale prices
						$this->import_wholesale_prices($role_slug, $group['customer_group_id']);
					}
				}
			}
			$this->plugin->display_admin_notice(sprintf(_n('%d role imported', '%d roles imported', $roles_count, __CLASS__), $roles_count));
		}
		
		/**
		 * Import the user groups
		 * 
		 * @since 1.1.0
		 * 
		 * @param int $user_id User ID
		 * @param array $customer Customer data
		 */
		public function import_customer_groups($user_id, $customer) {
			if ( $this->customer_groups_options['import_groups'] == 'as_groups' ) {
				$this->import_customer_groups_as_groups($user_id, $customer);
			} else {
				$this->import_customer_groups_as_roles($user_id, $customer);
			}
		}
		
		/**
		 * Assign groups to the customer
		 * 
		 * @param int $user_id User ID
		 * @param array $customer Customer data
		 * @uses Groups_User_Group (Groups plugin)
		 */
		private function import_customer_groups_as_groups($user_id, $customer) {
			if ( class_exists('Groups_Group') && class_exists('Groups_User_Group') ) {
				$this->remove_user_groups($user_id); // Remove the Registered group which is created by default
				if ( !empty($customer['group_id']) ) {
					$group_id = $this->imported_groups[$customer['group_id']];
					Groups_User_Group::create(array(
						'user_id'	=> $user_id,
						'group_id'	=> $group_id,
					));
				}
			}
		}
		
		/**
		 * Assign roles to the customer
		 * 
		 * @since 1.1.0
		 * 
		 * @param int $user_id WordPress user ID
		 * @param array $customer Customer data
		 */
		public function import_customer_groups_as_roles($user_id, $customer) {
			if ( isset($this->imported_roles[$customer['group_id']]) ) {
				$user = get_user_by('id', $user_id);
				$user->add_role($this->imported_roles[$customer['group_id']]);
			}
		}
		
		/**
		 * Remove all the groups from a user
		 * 
		 * @param int $user_id User ID
		 */
		private function remove_user_groups($user_id) {
			global $wpdb;
			
			$user_group_table = $wpdb->prefix . 'groups_user_group';
			$sql = $wpdb->prepare("DELETE FROM $user_group_table WHERE user_id = %d", $user_id);
			$wpdb->query($sql);
		}
		
		/**
		 * Get all the User Groups
		 * 
		 * @return array Array of groups
		 */
		private function get_groups() {
			$groups = array();
			$prefix = $this->plugin->plugin_options['prefix'];
			
			$sql = "
				SELECT customer_group_id, customer_group_code
				FROM ${prefix}customer_group
			";
			$groups = $this->plugin->magento_query($sql);
			return $groups;
		}
		
		/**
		 * Import the wholesale prices
		 * 
		 * @since 1.2.0
		 * 
		 * @param string $role_slug Role slug
		 * @param int $customer_group_id Customer group ID
		 */
		private function import_wholesale_prices($role_slug, $customer_group_id) {
			$tier_prices = $this->get_tier_prices($customer_group_id);
			if ( !empty($tier_prices) ) {
				$this->plugin->display_admin_notice(sprintf(__('Importing the %s wholesale prices...', __CLASS__), $role_slug));
			
				foreach ( $tier_prices as $tier_price ) {
					// Import the product wholesale price
					foreach ( $this->imported_products as $imported_products ) {
						if ( array_key_exists($tier_price['entity_id'], $imported_products) ) {
							$this->import_wholesale_price($role_slug, $tier_price['value'], $tier_price['percentage_value'], $imported_products[$tier_price['entity_id']]);
						}
					}
				}
			}
		}
		
		/**
		 * Get the tier prices of a customer group
		 * 
		 * @since 1.2.0
		 * 
		 * @param int $customer_group_id Customer group ID
		 * @return array Tier prices
		 */
		private function get_tier_prices($customer_group_id) {
			$prices = array();
			$prefix = $this->plugin->plugin_options['prefix'];
			
			$table = 'catalog_product_entity_tier_price';
			if ( version_compare($this->plugin->magento_version, '2', '<') ) {
				// Magento 1
				$percentage_value_field = '"" AS percentage_value';
				if ( $this->customer_groups_options['import_group_prices'] == 'from_group_prices' ) {
					$table = 'catalog_product_entity_group_price'; // Group_price table
				}
			} else {
				// Magento 2
				$percentage_value_field = 'tp.percentage_value';
			}
			$sql = "
				SELECT tp.entity_id, tp.value, $percentage_value_field
				FROM ${prefix}$table tp
				WHERE tp.customer_group_id = '$customer_group_id'
			";
			$prices = $this->plugin->magento_query($sql);
			
			return $prices;
		}
		
		/**
		 * Import a wholesale price for a product or for a variation
		 * 
		 * @since 1.2.0
		 * 
		 * @param string $role_slug Role slug
		 * @param float $fixed_price Fixed price
		 * @param float $percentage_reduction Reduction percentage
		 * @param int $post_id Product ID or Variation ID
		 */
		private function import_wholesale_price($role_slug, $fixed_price, $percentage_reduction, $post_id) {
			$wholesale_price = 0.0;
			if ( $fixed_price != 0.0 ) {
				$wholesale_price = $fixed_price;
			} else {
				$price = floatval(get_post_meta($post_id, '_regular_price', true));
				if ( ($price != 0.0) && !empty($percentage_reduction) ) {
					$wholesale_price = $price * $percentage_reduction / 100;
				}
			}
			if ( $wholesale_price != 0.0 ) {
				add_post_meta($post_id, $role_slug . '_have_wholesale_price', 'yes', true);
				add_post_meta($post_id, $role_slug . '_wholesale_price', $wholesale_price, true);
			}
		}
		
	}
}
